<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Exports\SuperAdmin\TherapistEducationsExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\TherapistCertifications\CreateRequest;
use App\Http\Requests\ImportRequest;
use App\Imports\SuperAdmin\TherapistEducationsImport;
use App\Models\Certification;
use App\Models\Therapist;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;

class TherapistCertificationsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    /********* Initialize Permission based Middlewares  ***********/
    public function __construct()
    {
        $this->middleware('permission:therapist.add_certification');
        $this->middleware('permission:therapist.add_certification', ['only' => ['store']]);
        $this->middleware('permission:therapist.add_certification', ['only' => ['update']]);
        $this->middleware('permission:therapist.add_certification', ['only' => ['destroy']]);
        $this->middleware('permission:therapist.add_certification', ['only' => ['export']]);
        $this->middleware('permission:therapist.add_certification', ['only' => ['import']]);
    }
    /********* Getter For Pagination, Searching And Sorting  ***********/
    public function getter($req = null, $export = null, $therapist)
    {
        if ($req != null) {
            $therapist_certifications =  $therapist->therapist_certifications();
            if ($req->trash && $req->trash == 'with') {
                $therapist_certifications =  $therapist_certifications->withTrashed();
            }
            if ($req->trash && $req->trash == 'only') {
                $therapist_certifications =  $therapist_certifications->onlyTrashed();
            }
            if ($req->column && $req->column != null && $req->search != null) {
                $therapist_certifications = $therapist_certifications->whereLike($req->column, $req->search);
            } else if ($req->search && $req->search != null) {

                $therapist_certifications = $therapist_certifications->whereLike(['name', 'description'], $req->search);
            }
            if ($req->sort_field != null && $req->sort_type != null) {
                $therapist_certifications = $therapist_certifications->OrderBy($req->sort_field, $req->sort_type);
            } else {
                $therapist_certifications = $therapist_certifications->OrderBy('id', 'desc');
            }
            if ($export != null) { // for export do not paginate
                $therapist_certifications = $therapist_certifications->get();
                return $therapist_certifications;
            }
            $therapist_certifications = $therapist_certifications->get();
            return $therapist_certifications;
        }
        $therapist_certifications = $therapist->therapist_certifications()->withAll()->orderBy('id', 'desc')->get();
        return $therapist_certifications;
    }


    /*********View All TherapistCertifications  ***********/
    public function index(Request $request, Therapist $therapist)
    {
        $therapist_certifications = $this->getter($request, null, $therapist);
        return view('super_admins.therapists.therapist_certifications.index', compact('therapist_certifications', 'therapist'));
    }

    /*********View Create Form of Certification  ***********/
    public function create(Therapist $therapist)
    {
        return view('super_admins.therapists.therapist_certifications.create', compact('therapist'));
    }

    /*********Store Certification  ***********/
    public function store(CreateRequest $request, Therapist $therapist)
    {
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $data = $request->all();
            $data['image'] = uploadFile($request, 'file', 'therapist_certifications');
            $therapist_certification = $therapist->therapist_certifications()->create($data);
            $therapist_certification = $therapist->therapist_certifications()->withAll()->find($therapist_certification->id);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_certifications.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_certifications.index', $therapist->id)->with('message', 'Certificate Created Successfully')->with('message_type', 'success');
    }

    /*********View Certification  ***********/
    public function show(Therapist $therapist, Certification $therapist_certification)
    {
        if ($therapist->id != $therapist_certification->therapist_id) {
            return redirect()->back()->with('message', 'Certification Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapists.therapist_certifications.show', compact('therapist_certification', 'therapist'));
    }

    /*********View Edit Form of Certification  ***********/
    public function edit(Therapist $therapist, Certification $therapist_certification)
    {
        if ($therapist->id != $therapist_certification->therapist_id) {
            return redirect()->back()->with('message', 'Certification Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapists.therapist_certifications.edit', compact('therapist_certification', 'therapist'));
    }

    /*********Update Certification  ***********/
    public function update(CreateRequest $request, Therapist $therapist, Certification $therapist_certification)
    {
        if ($therapist->id != $therapist_certification->therapist_id) {
            return redirect()->back()->with('message', 'Certification Not Found')->with('message_type', 'error');
        }
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $data = $request->all();
            if ($request->file) {
                $data['image'] = uploadFile($request, 'file', 'therapist_certifications', $therapist_certification->image);
            } else {
                $data['image'] = $therapist_certification->image;
            }
            $therapist_certification->update($data);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_certifications.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_certifications.index', $therapist->id)->with('message', 'Certification Updated Successfully')->with('message_type', 'success');
    }

    /********* Export  CSV And Excel  **********/
    public function export(Request $request)
    {
        $therapist_certifications = $this->getter($request, "export");
        if (in_array($request->export, ['csv,xlsx'])) {
            $extension = $request->export;
        } else {
            $extension = 'xlsx';
        }
        $filename = "therapist_certifications." . $extension;
        return Excel::download(new TherapistEducationsExport($therapist_certifications), $filename);
    }
    /********* Import CSV And Excel  **********/
    public function import(ImportRequest $request)
    {
        $file = $request->file('file');
        Excel::import(new TherapistEducationsImport, $file);
        return redirect()->back()->with('message', 'Certification Categories imported Successfully')->with('message_type', 'success');
    }


    /*********Soft DELETE Certification ***********/
    public function destroy(Therapist $therapist, Certification $therapist_certification)
    {
        if ($therapist->id != $therapist_certification->therapist_id) {
            return redirect()->back()->with('message', 'Certification Not Found')->with('message_type', 'error');
        }
        $therapist_certification->delete();
        return redirect()->back()->with('message', 'Certification Deleted Successfully')->with('message_type', 'success');
    }

    /*********Permanently DELETE Certification ***********/
    public function destroyPermanently(Request $request, Therapist $therapist, $therapist_certification)
    {
        $therapist_certification = Certification::withTrashed()->find($therapist_certification);
        if ($therapist_certification) {
            if ($therapist_certification->trashed()) {
                if ($therapist_certification->image && file_exists(public_path($therapist_certification->image))) {
                    unlink(public_path($therapist_certification->image));
                }
                $therapist_certification->forceDelete();
                return redirect()->back()->with('message', 'Certification Deleted Successfully')->with('message_type', 'success');
            } else {
                return redirect()->back()->with('message', 'Certification is Not in Trash')->with('message_type', 'error');
            }
        } else {
            return redirect()->back()->with('message', 'Certification Not Found')->with('message_type', 'error');
        }
    }
    /********* Restore Certification***********/
    public function restore(Request $request, Therapist $therapist, $therapist_certification)
    {
        $therapist_certification = Certification::withTrashed()->find($therapist_certification);
        if ($therapist_certification->trashed()) {
            $therapist_certification->restore();
            return redirect()->back()->with('message', 'Certification Restored Successfully')->with('message_type', 'success');
        } else {
            return redirect()->back()->with('message', 'Certification Not Found')->with('message_type', 'error');
        }
    }
}
